function [colo_data] = colo(varargin)
%This function is more like a script that helps the user analyze their data
%for colocalization.  Difference/sum of the image correlations.
%Synatax:   [colo_data] = colo('x',x,'y',y,'maxoffset',10); or
%           [colo_data] = colo([x y],'maxoffset',10);
%Input:     This function does not really require any inputs.  However if
%           you want to specify the point in the data set to choose for
%           your colocalization analysis.
%           x = x coordinate
%           y = y coordinate
%           [x y] = Specify the x y  by a 1x2 array, where 1 = x and 2 = y
%           maxoffset = parameter will determine the area used for
%           colocalization analysis.  Default = [10, 10, 14].
%
%Output:    colo_data = a structured array with the data for co-localization
%           and the coordinates used for the analysis.

[vertex,maxoffset,verbose] = parse(varargin);     %Parse the input for the vertex
optsize=2*maxoffset*10+10;                %This is the area used for the co localization analysis

%import files as stacks, verify that the stack dimensions are identical, if
%not give a warning.
%open the base stack.  Prompt first
prompt_box('title','Open Base Channel','prompt1','Select the channel you want to compare other',...
    'prompt2','channels against','position','center');
[stks,img_range,channel_names,channel_paths] = stack_gui(1);  %use cell output mode.
channel_paths = {channel_paths};    %make the path a cell array to more succinct data storage

%now select all the channels you want to compare the base channel against
prompt_box('title','Open Comparision Channels','prompt1','Select the channels you want to use for',...
    'prompt2','colocalization analysis.','position','center');
pick_ch = 0;    %initialize channel picking switch
while pick_ch==0
    [stk_tmp,img_range_tmp,channel_names_tmp,channel_paths_tmp] = stack_gui(1);   %open the channels
    stks = cat(2,stks,stk_tmp);     %join the channels together in a single cell array
    img_range = cat(2,img_range,img_range_tmp);
    channel_names = cat(2,channel_names,channel_names_tmp);
    channel_paths = cat(2,channel_paths,channel_paths_tmp);
    %clear the stk_tmp to free memory, and if the user want to select another stack
    clear stk_tmp img_range_tmp channel_names_tmp channel_paths_tmp; 
    %ask the user if they want to continue
    pick_ch = yes_no_box('title','Are you done selecting channels?','caption1','Select yes is you want to move on.',...
        'caption2','Select no if you want to import another channel.','position','center');
end

%store some of the file attribes in out data structure
colo_data(1).img_range = img_range;
colo_data(1).channel_names = channel_names;
colo_data(1).channel_paths = channel_paths;

%error check make sure the vertex selected, if selected is in the image,
%and make sure the images are of equivalent sizes.  Also, check to make
%sure that the image is large enough for the selected area of analysis, and
%that the vertex selected allows enough room in the image for the anlysis.
if optsize(2)>size(stks{1,1},1)||optsize(1)>size(stks{1,1},2)     %image large enough
    error('The image you provided is too small.  Import a larger image or change the maxoffset.')
end
if ~isempty(vertex)
    if vertex(1,1)<ceil(optsize(2)/2)||vertex(1,1)>size(stks{1,1},2)-ceil(optsize(2)/2)...        %Vertex placed with enough room.
            ||vertex(1,2)<ceil(optsize(1)/2)||vertex(1,1)>size(stks{1,1},1)-ceil(optsize(1)/2)
        warning('The vertex you selected is too close to the image borders.  Revert to Default.');
        vertex = [];
    end
end

for i = 1:size(stks,2)
    stk_sz{1,i} = size(stks{1,i});      %get the size of each stack
    if i>1                              %compare size
        if stk_sz{1,i-1}~=stk_sz{1,i}
            warning('The image stacks you opened are not of the same dimensions.  This could be problemmatic.');
        end
    end
end
        

%Now pick the region for the colocalization analysis. If vertex is empty.
if isempty(vertex)
    warning off all;
    img = flatview(stks{1,1},'mean');    %create the image for selection based on the base channel
    warning on all;
    h = figure('units','pixels','name','Double click to select the point of reference');  %,'menubar','none','toolbar','none'); %for reference
    %Note: For future reference, You can get the actual screen size by the command: get(0,'screensize')    
    colo_data(1).img=img;
    imshow(img/max(max(img)));                                         %Display Image    
    %colormap(gray);                                     %Grayscale image
    uconfirm = 0;                                       %Initialize
    while uconfirm==0                                   %let user confirm that the box is correct
        p = impoint(gca,[size(img,2)/2 size(img,1)/2]);     %allow the user to select a point
        vertex = wait(p);                                   %wait for the point selection
        vertex = round(vertex);                             %use pixel coordinates for now, subpixel another day
        h_rec = rectangle('Position',[(floor(vertex(1,1))-optsize(2)/2),(floor(vertex(1,2))-optsize(1)/2),optsize(2),optsize(1)],...
            'LineWidth',2,'EdgeColor',[1 0 1]);    %Draw the colocalization area
        uconfirm = yes_no_box('title','Is the point at the right location?','caption1','Select yes is you want to move on.',...
            'caption2','Select no if you want to reposition the point','position','west');
        if uconfirm==0
            delete(p);
            delete(h_rec);
        end
    end
    delete(h);        %close figure1
end
colo_data(1).vertex = vertex;   %place vertex into the output struct array
colo_data(1).maxoffset = maxoffset; %place maxoffset value in the output

%Now run the colocalization analysis.
[meanimage,stderrimage,images,rotimages,corr_img] = ImageCorrN(stks,maxoffset,vertex,channel_names);
%store the results in our structure.
colo_data(1).meanimage = meanimage;
colo_data(1).stderrimage = stderrimage;
colo_data(1).images = images;
colo_data(1).rotimages = rotimages;
colo_data(1).corr_img = corr_img;

%Now quantify the data
[firstprofs,stdprofs,analyzed_img] = GetProfiles2(images(:,:,floor(size(images,3)/2)+1),maxoffset(1),stderrimage);
%Store the results
colo_data(1).firstprofs = firstprofs;
colo_data(1).stdprofs = stdprofs;
colo_data(1).analyzed_img = analyzed_img;

%Ask if you want to save the data into excel form.
uconfirm = yes_no_box('title','Do you want to save the data?','caption1','Select yes to save as an Excel file',...
            'caption2','Select no if you are not interested','position','center');
%convert firstprof and stdprof to something that can be exported to Excel
%in a readable form.
z = 1;  %initialize an external counter
for i = 1:size(firstprofs,1)
    for j = 1:size(firstprofs,2)
        profiles_data(z,:) = firstprofs{i,j};   %write down 1 cell of firstprofs
        firstprofs_m(j,:,i) = firstprofs{i,j};  %create a matrix for easy plotting in matlab.  z=each channel
        z = z+1;                                %step down the line one step
        profiles_data(z,:) = stdprofs{i,j};     %write-down the corresponding stdprofs cell
        stdprofs_m(j,:,i) = stdprofs{i,j};      %create a matrix for easy plotting in matlab.
        z = z+1;                                %step down to the next line
    end
    z = z+1;                                    %leave 1 empty space between each row of cells.  Empty = 0
end
colo_data(1).profiles_data = profiles_data;     %store the profiles array
colo_data(1).firstprofs_m = firstprofs_m;
colo_data(1).stdprofs_m = stdprofs_m;               
xls_exp = dataset(profiles_data);               %to export as a XLS need to make the data a dataset.
if uconfirm==1      %save data as an excel file
    [filename,pathname,filterindex] = uiputfile('.xls');
    export(xls_exp,'XLSfile',[pathname,filename]);
end
    
%plot the data for immediate gratification.
if verbose
    for i = 1:size(firstprofs_m,3)        %plot for each channel.
        figure;                           %new figure.
        h = plot(firstprofs_m(:,:,i)');   %plot the graph.
        %errorbar(size(firstprofs_m,2),stdprofs_m(:,:,i)')   %draw errorbars
        title(gca,channel_names{i});          %title the graph.
        legend(h,channel_names);            %put up a legend
    end
end

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [vertex,maxoffset,verbose] = parse(input)

vertex = [];    %initialize-if vertex is empty at output = default
maxoffset = [10,10,7];  %Default Initialized.
verbose = 0;        %no output of graphs by default

%Parse the input
if ~isempty(input)
    for i = 1:size(input,2)
        if isnumeric(input{1,i})&&size(input{1,i},2)==2    %input is [x y]: Process...
            vertex = input{1,1};
        elseif ischar(input{1,i});
            switch input{1,i}
                case 'x'
                    vertex(1,1) = input{1,i+1};
                case 'y'
                    vertex(1,2) = input{1,i+1};
                case 'maxoffset'
                    maxoffset = input{1,i+1};
                case 'verbose'
                    verbose = 1;
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        end
    end
end
        